// settings.js
document.addEventListener('DOMContentLoaded', async function() {
  let currentUser = null;
  let isLoading = false;

  // Configuration
  const postTypes = [
    { id: 'linkedin', name: 'LinkedIn Post', description: 'Professional updates' },
    { id: 'twitter', name: 'Twitter Thread', description: 'Quick insights' },
    { id: 'blog', name: 'Blog Post', description: 'Detailed articles' },
    { id: 'all', name: 'All Formats', description: 'Multi-platform content' }
  ];

  const tones = [
    { id: 'professional', name: 'Professional', description: 'Formal and business-focused' },
    { id: 'casual', name: 'Casual', description: 'Friendly and conversational' },
    { id: 'inspiring', name: 'Inspiring', description: 'Motivational and uplifting' }
  ];

  // DOM Elements
  const backBtn = document.getElementById('back-btn');
  const saveBtn = document.getElementById('save-btn');
  const saveText = document.getElementById('save-text');
  const saveLoading = document.getElementById('save-loading');
  const saveMessage = document.getElementById('save-message');

  // Initialize the page
  await initializePage();

  // Event Listeners
  backBtn.addEventListener('click', function() {
    window.close();
  });

  saveBtn.addEventListener('click', handleSave);

  // LinkedIn input validation
  document.getElementById('linkedin').addEventListener('input', function(e) {
    const tip = document.getElementById('linkedin-tip');
    tip.style.display = e.target.value && !e.target.value.includes('linkedin.com') ? 'block' : 'none';
  });

  // Twitter input validation
  document.getElementById('twitter').addEventListener('input', function(e) {
    const tip = document.getElementById('twitter-tip');
    tip.style.display = e.target.value && !e.target.value.startsWith('@') ? 'block' : 'none';
  });

  async function initializePage() {
    try {
      // Get user data from Chrome storage
      const result = await chrome.storage.local.get(['user']);
      
      if (result.user) {
        currentUser = result.user;
        populateForm(currentUser);
        createPostTypeOptions();
        createToneOptions();
      } else {
        // Try to fetch user from backend if not in storage
        await fetchUserData();
      }
    } catch (error) {
      console.error('Error initializing settings:', error);
      showMessage('Failed to load user data', 'error');
    }
  }

  async function fetchUserData() {
    try {
      // Send message to background script to fetch user data
      const response = await chrome.runtime.sendMessage({ 
        action: 'FETCH_USER',
        userId: currentUser?.id 
      });

      if (response.success) {
        currentUser = response.user;
        populateForm(currentUser);
        createPostTypeOptions();
        createToneOptions();
      }
    } catch (error) {
      console.error('Error fetching user data:', error);
    }
  }

  function populateForm(user) {
    // Set avatar
    const avatarContainer = document.getElementById('avatar-container');
    if (user.picture) {
      avatarContainer.innerHTML = `<img src="${user.picture}" alt="User Avatar" class="user-avatar">`;
    } else {
      const initial = user.name ? user.name.charAt(0).toUpperCase() : 'U';
      avatarContainer.innerHTML = `<div class="avatar-fallback">${initial}</div>`;
    }

    // Set form values
    document.getElementById('name').value = user.name || '';
    document.getElementById('email').value = user.email || '';
    document.getElementById('skillFocus').value = user.preferences?.skillFocus || '';

    // Social handles
    document.getElementById('linkedin').value = user.preferences?.linkedin_handle || '';
    document.getElementById('twitter').value = user.preferences?.twitter_handle || '';

    // Set post type
    const postType = user.preferences?.postType || 'linkedin';
    document.querySelectorAll('.option-card').forEach(card => {
      card.classList.remove('selected');
      if (card.dataset.value === postType) {
        card.classList.add('selected');
      }
    });

    // Set tone
    const tone = user.preferences?.tone || 'professional';
    document.querySelectorAll('.tone-option').forEach(option => {
      option.classList.remove('selected');
      if (option.dataset.value === tone) {
        option.classList.add('selected');
      }
    });

    // Set share progress toggle
    const shareProgress = user.preferences?.shareProgress !== false;
    const toggle = document.getElementById('share-progress-toggle');
    toggle.classList.toggle('on', shareProgress);
  }

  function createPostTypeOptions() {
    const container = document.getElementById('post-type-options');
    container.innerHTML = '';

    postTypes.forEach(type => {
      const card = document.createElement('div');
      card.className = 'option-card';
      card.dataset.value = type.id;
      card.innerHTML = `
        <div class="option-name">${type.name}</div>
        <div class="option-desc">${type.description}</div>
      `;
      
      card.addEventListener('click', () => {
        document.querySelectorAll('.option-card').forEach(c => c.classList.remove('selected'));
        card.classList.add('selected');
      });

      container.appendChild(card);
    });
  }

  function createToneOptions() {
    const container = document.getElementById('tone-options');
    container.innerHTML = '';

    tones.forEach(tone => {
      const option = document.createElement('div');
      option.className = 'tone-option';
      option.dataset.value = tone.id;
      option.innerHTML = `
        <div class="tone-info">
          <h3>${tone.name}</h3>
          <p>${tone.description}</p>
        </div>
        <div class="radio-dot"></div>
      `;
      
      option.addEventListener('click', () => {
        document.querySelectorAll('.tone-option').forEach(o => o.classList.remove('selected'));
        option.classList.add('selected');
      });

      container.appendChild(option);
    });
  }

  async function handleSave() {
    if (isLoading) return;

    try {
      isLoading = true;
      setLoading(true);

      const formData = {
        name: document.getElementById('name').value,
        email: document.getElementById('email').value,
        skillFocus: document.getElementById('skillFocus').value,
        linkedin_handle: document.getElementById('linkedin').value,
        twitter_handle: document.getElementById('twitter').value,
        postType: document.querySelector('.option-card.selected')?.dataset.value || 'linkedin',
        tone: document.querySelector('.tone-option.selected')?.dataset.value || 'professional',
        shareProgress: document.getElementById('share-progress-toggle').classList.contains('on')
      };

      // Send update to background script
      const response = await chrome.runtime.sendMessage({
        action: 'UPDATE_USER',
        userData: {
          id: currentUser.id,
          ...formData
        }
      });

      if (response.success) {
        showMessage('Settings saved successfully!', 'success');
        // Update local user data
        currentUser = response.user;
        await chrome.storage.local.set({ user: response.user });
      } else {
        throw new Error(response.error || 'Failed to save settings');
      }

    } catch (error) {
      console.error('Error saving settings:', error);
      showMessage(error.message || 'Failed to save settings', 'error');
    } finally {
      isLoading = false;
      setLoading(false);
    }
  }

  function setLoading(loading) {
    if (loading) {
      saveText.style.display = 'none';
      saveLoading.style.display = 'inline-block';
      saveBtn.disabled = true;
    } else {
      saveText.style.display = 'inline-block';
      saveLoading.style.display = 'none';
      saveBtn.disabled = false;
    }
  }

  function showMessage(message, type) {
    saveMessage.textContent = message;
    saveMessage.className = type;
    saveMessage.style.display = 'block';

    setTimeout(() => {
      saveMessage.style.display = 'none';
    }, 3000);
  }

  // Toggle switch functionality
  document.getElementById('share-progress-toggle').addEventListener('click', function() {
    this.classList.toggle('on');
  });
});