let screenshots = [];
let currentModalIndex = 0;
let postContent = '';

const API_BASE_URL = 'https://hudddle-post-feature.onrender.com';
const LOCAL_API_URL = 'http://127.0.0.1:8000';

document.addEventListener('DOMContentLoaded', async function() {
  const urlParams = new URLSearchParams(window.location.search);
  const encodedPost = urlParams.get('post');
  const userId = urlParams.get('userId');
  const sessionId = urlParams.get('sessionId');
  
  const postContentElement = document.getElementById('post-content');
  const copyBtn = document.getElementById('copy-btn');
  const backBtn = document.getElementById('back-btn');
  const shareLinkedIn = document.getElementById('share-linkedin');
  const shareTwitter = document.getElementById('share-twitter');
  
  // Load post content
  if (encodedPost) {
    postContent = decodeURIComponent(encodedPost);
    postContentElement.textContent = postContent;
    
    // Show share buttons if post is loaded
    shareLinkedIn.style.display = 'inline-flex';
    shareTwitter.style.display = 'inline-flex';
  } else {
    postContentElement.innerHTML = `
      <div class="empty-state">
        <div style="font-size: 48px; margin-bottom: 16px;">📝</div>
        <p>No post content available.</p>
      </div>
    `;
  }
  
  // Load screenshots if userId is provided
  if (userId) {
    await loadScreenshots(userId);
  }
  
  // Event listeners
  copyBtn.addEventListener('click', copyPost);
  backBtn.addEventListener('click', () => window.close());
  shareLinkedIn.addEventListener('click', shareOnLinkedIn);
  shareTwitter.addEventListener('click', shareOnTwitter);
  
  setupModalHandlers();
});

async function loadScreenshots(userId) {
  try {
    const screenshotsSection = document.getElementById('screenshots-section');
    const screenshotsGrid = document.getElementById('screenshots-grid');
    const screenshotCount = document.getElementById('screenshot-count');
    
    // Show loading state
    screenshotsGrid.innerHTML = `
      <div style="grid-column: 1 / -1;" class="loading">
        <div class="spinner"></div>
        Loading screenshots...
      </div>
    `;
    screenshotsSection.style.display = 'block';

    // Get sessionId from URL params
    const urlParams = new URLSearchParams(window.location.search);
    const sessionId = urlParams.get('sessionId');
    
    let response;
    
    if (!sessionId || sessionId === 'latest') {
      // Fetch screenshots from the most recent session
      response = await fetch(`${API_BASE_URL}/user/${userId}/screenshots/latest`);
    } else {
      // Fetch screenshots for the specific session
      response = await fetch(`${API_BASE_URL}/user/${userId}/session/${sessionId}/screenshots`);
    }
    
    if (!response.ok) {
      throw new Error(`Failed to fetch screenshots: ${response.status}`);
    }
    
    const data = await response.json();
    screenshots = data.screenshots || [];
    
    if (screenshots.length > 0) {
      screenshotCount.textContent = `${screenshots.length} images`;
      renderScreenshots();
    } else {
      screenshotsGrid.innerHTML = `
        <div style="grid-column: 1 / -1;" class="empty-state">
          <div style="font-size: 48px; margin-bottom: 16px;">📷</div>
          <p>No screenshots available for this user.</p>
        </div>
      `;
    }
    
  } catch (error) {
    console.error('Failed to load screenshots:', error);
    const screenshotsGrid = document.getElementById('screenshots-grid');
    screenshotsGrid.innerHTML = `
      <div style="grid-column: 1 / -1;" class="empty-state">
        <div style="font-size: 48px; margin-bottom: 16px;">⚠️</div>
        <p>Failed to load screenshots: ${error.message}</p>
      </div>
    `;
  }
}

function renderScreenshots() {
  const screenshotsGrid = document.getElementById('screenshots-grid');
  
  screenshotsGrid.innerHTML = screenshots.map((screenshot, index) => {
    const date = new Date(screenshot.created_at);
    const timeStr = date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
    
    return `
      <div class="screenshot-item" data-index="${index}" onclick="openModal(${index})">
        <img src="${screenshot.url}" alt="Screenshot ${index + 1}" loading="lazy">
        <div class="screenshot-overlay">
          <div>Screenshot ${index + 1}</div>
          <div>${timeStr}</div>
        </div>
      </div>
    `;
  }).join('');
}

function setupModalHandlers() {
  const modal = document.getElementById('screenshot-modal');
  const modalClose = document.getElementById('modal-close');
  const modalPrev = document.getElementById('modal-prev');
  const modalNext = document.getElementById('modal-next');
  
  modalClose.addEventListener('click', closeModal);
  modalPrev.addEventListener('click', () => navigateModal(-1));
  modalNext.addEventListener('click', () => navigateModal(1));
  
  // Close modal when clicking outside
  modal.addEventListener('click', (e) => {
    if (e.target === modal) {
      closeModal();
    }
  });
  
  // Keyboard navigation
  document.addEventListener('keydown', (e) => {
    if (!modal.classList.contains('active')) return;
    
    switch (e.key) {
      case 'Escape':
        closeModal();
        break;
      case 'ArrowLeft':
        navigateModal(-1);
        break;
      case 'ArrowRight':
        navigateModal(1);
        break;
    }
  });
}

function openModal(index) {
  if (!screenshots[index]) return;
  
  currentModalIndex = index;
  const modal = document.getElementById('screenshot-modal');
  const modalImage = document.getElementById('modal-image');
  const modalPrev = document.getElementById('modal-prev');
  const modalNext = document.getElementById('modal-next');
  
  modalImage.src = screenshots[index].url;
  modalImage.alt = `Screenshot ${index + 1}`;
  modal.classList.add('active');
  
  // Show/hide navigation buttons
  modalPrev.style.display = index > 0 ? 'flex' : 'none';
  modalNext.style.display = index < screenshots.length - 1 ? 'flex' : 'none';
  
  // Prevent body scroll
  document.body.style.overflow = 'hidden';
}

function closeModal() {
  const modal = document.getElementById('screenshot-modal');
  modal.classList.remove('active');
  document.body.style.overflow = '';
}

function navigateModal(direction) {
  const newIndex = currentModalIndex + direction;
  if (newIndex >= 0 && newIndex < screenshots.length) {
    openModal(newIndex);
  }
}

async function copyPost() {
  const copyBtn = document.getElementById('copy-btn');
  try {
    await navigator.clipboard.writeText(postContent);
    const originalText = copyBtn.textContent;
    copyBtn.textContent = '✅ Copied!';
    copyBtn.style.background = '#10b981';
    
    setTimeout(() => {
      copyBtn.textContent = originalText;
      copyBtn.style.background = '#667eea';
    }, 2000);
  } catch (error) {
    console.error('Failed to copy:', error);
    alert('Failed to copy post. Please select and copy manually.');
  }
}

function shareOnLinkedIn() {
  const text = encodeURIComponent(postContent);
  const url = `https://www.linkedin.com/sharing/share-offsite/?url=${encodeURIComponent(window.location.href)}&text=${text}`;
  window.open(url, '_blank', 'width=600,height=400');
}

function shareOnTwitter() {
  const text = encodeURIComponent(postContent.substring(0, 280)); // Twitter character limit
  const url = `https://x.com/intent/tweet?text=${text}`;
  window.open(url, '_blank', 'width=600,height=400');
}
