// Popup script for Hudddle Extension
let currentUser = null;
let sessionActive = false;
let sessionStartTime = null;
let timeUpdateInterval = null;
let screenshotCount = 0;

// DOM elements - will be assigned when sections are created
let googleAuthBtn, startBtn, stopBtn, settingsBtn, onboardingBtn;
let authSection, onboardingSection, sessionSection, loadingIndicator;
let userNameDisplay, todaySessions, streakDisplay, screenshotCountDisplay;
let statusIndicator, timeDisplay;

// Initialize popup
document.addEventListener('DOMContentLoaded', async () => {
  await initializePopup();
});

async function initializePopup() {
  try {
    showLoading(true);

    // Check session status first to get any active session
    const sessionResponse = await chrome.runtime.sendMessage({ action: 'GET_SESSION_STATUS' });
    
    if (sessionResponse.active) {
      sessionActive = true;
      sessionStartTime = sessionResponse.startTime;
      screenshotCount = sessionResponse.screenshotCount || 0;
      if (sessionResponse.user) {
        currentUser = sessionResponse.user;
      }
    }
    
    // Check authentication status first
    const authResponse = await chrome.runtime.sendMessage({ action: 'CHECK_AUTH' });
    
    if (authResponse.success && authResponse.isAuthenticated) {
      if (!currentUser) {
        currentUser = authResponse.user;
      }
      await checkUserOnboardingStatus();
    } else {
      showAuthSection();
    }
    
    showLoading(false);
  } catch (error) {
    console.error('❌ Error initializing popup:', error);
    showAuthSection();
    showLoading(false);
  }
}

function showLoading(show) {
  const container = document.getElementById('app-container');
  if (show) {
    container.innerHTML = `
      <div class="loading-container">
        <div class="spinner"></div>
        <p>Loading...</p>
      </div>
    `;
  }
}

async function checkUserOnboardingStatus() {
  if (!currentUser) {
    showAuthSection();
    return;
  }

  // Fetch latest user data from backend
  try {
    const userResponse = await chrome.runtime.sendMessage({
      action: 'FETCH_USER',
      userId: currentUser.id
    });
    
    if (userResponse.success) {
      currentUser = userResponse.user;
      
      if (currentUser.preferences && Object.keys(currentUser.preferences).length > 0) {
        showSessionSection();
      } else {
        showOnboardingSection();
      }
    } else {
      console.error('Failed to fetch user data:', userResponse.error);
      if (userResponse.error.includes('Authentication expired')) {
        showAuthSection();
      } else {
        showOnboardingSection(); // Default to onboarding if fetch fails
      }
    }
  } catch (error) {
    console.error('Error fetching user data:', error);
    showOnboardingSection();
  }
}

async function checkSessionStatus() {
  try {
    const response = await chrome.runtime.sendMessage({ action: 'GET_SESSION_STATUS' });
    
    if (response.active) {
      sessionActive = true;
      sessionStartTime = response.startTime;
      screenshotCount = response.screenshotCount || 0;

      if (response.user && !currentUser) {
        currentUser = response.user;
      }
      
      if (sessionSection && !sessionSection.classList.contains('hidden')) {
        showActiveSession();
        startTimeUpdater();
      }
    } else {
      sessionActive = false;
      if (sessionSection && !sessionSection.classList.contains('hidden')) {
        showIdleSession();
      }
    }
  } catch (error) {
    console.error('❌ Error checking session status:', error);
  }
}

function showAuthSection() {
  const container = document.getElementById('app-container');
  container.innerHTML = `
    <div class="auth-section">
      <div class="logo">
        <img src="icons/icon48.png" alt="Hudddle Logo" onerror="this.style.display='none'">
        <h2>Welcome to Hudddle</h2>
        <p>Track your building sessions and share your progress</p>
      </div>
      
      <div class="auth-tabs">
        <button id="login-tab" class="auth-tab active">Sign In</button>
        <button id="register-tab" class="auth-tab">Sign Up</button>
      </div>
      
      <!-- Login Form -->
      <form id="login-form" class="auth-form">
        <div class="form-group">
          <input type="email" id="login-email" placeholder="Email" required>
        </div>
        <div class="form-group">
          <input type="password" id="login-password" placeholder="Password" required>
        </div>
        <button type="submit" id="login-btn" class="auth-btn">Sign In</button>
      </form>
      
      <!-- Register Form -->
      <form id="register-form" class="auth-form hidden">
        <div class="form-group">
          <input type="text" id="register-name" placeholder="Full Name" required>
        </div>
        <div class="form-group">
          <input type="email" id="register-email" placeholder="Email" required>
        </div>
        <div class="form-group">
          <input type="password" id="register-password" placeholder="Password" required>
        </div>
        <button type="submit" id="register-btn" class="auth-btn">Sign Up</button>
      </form>
    </div>
  `;

  // Add CSS for the forms
  const style = document.createElement('style');
  style.textContent = `
    .auth-tabs {
      display: flex;
      margin-bottom: 20px;
      border-bottom: 1px solid #e5e7eb;
    }
    
    .auth-tab {
      flex: 1;
      padding: 12px;
      border: none;
      background: none;
      cursor: pointer;
      font-weight: 500;
      color: #ffffff;
      border-bottom: 2px solid transparent;
    }
    
    .auth-tab.active {
      color: #667eea;
      border-bottom-color: #667eea;
    }
    
    .auth-form {
      display: flex;
      flex-direction: column;
      gap: 16px;
    }
    
    .form-group input {
      width: 100%;
      padding: 12px 16px;
      border: 1px solid #d1d5db;
      border-radius: 8px;
      font-size: 16px;
      box-sizing: border-box;
    }
    
    .form-group input:focus {
      outline: none;
      border-color: #667eea;
      box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    }
    
    .auth-btn {
      padding: 12px 24px;
      background: #667eea;
      color: white;
      border: none;
      border-radius: 8px;
      font-size: 16px;
      font-weight: 600;
      cursor: pointer;
      transition: background-color 0.2s;
    }
    
    .auth-btn:hover {
      background: #5a67d8;
    }
    
    .auth-btn:disabled {
      background: #9ca3af;
      cursor: not-allowed;
    }
    
    .hidden {
      display: none !important;
    }
  `;
  document.head.appendChild(style);

  // Add event listeners
  const loginTab = document.getElementById('login-tab');
  const registerTab = document.getElementById('register-tab');
  const loginForm = document.getElementById('login-form');
  const registerForm = document.getElementById('register-form');

  loginTab.addEventListener('click', () => {
    loginTab.classList.add('active');
    registerTab.classList.remove('active');
    loginForm.classList.remove('hidden');
    registerForm.classList.add('hidden');
  });

  registerTab.addEventListener('click', () => {
    registerTab.classList.add('active');
    loginTab.classList.remove('active');
    registerForm.classList.remove('hidden');
    loginForm.classList.add('hidden');
  });

  loginForm.addEventListener('submit', handleLogin);
  registerForm.addEventListener('submit', handleRegister);
}

function showOnboardingSection() {
  const container = document.getElementById('app-container');
  container.innerHTML = `
    <div class="onboarding-section">
      <div class="user-header">
        <img src="${currentUser?.picture || 'icons/icon48.png'}" alt="User Avatar" class="user-avatar">
        <div>
          <h3>${currentUser?.name || 'User'}</h3>
          <p class="user-email">${currentUser?.email || ''}</p>
        </div>
      </div>
      
      <div class="onboarding-content">
        <h2>Complete Your Setup</h2>
        <p>Set up your preferences to get personalized content generation</p>
        
        <button id="onboarding-btn" class="action-btn primary">
          Complete Setup
        </button>
        
        <button id="skip-btn" class="action-btn secondary">
          Skip for Now
        </button>
      </div>
    </div>
  `;

  // Assign DOM elements and add event listeners
  onboardingBtn = document.getElementById('onboarding-btn');
  const skipBtn = document.getElementById('skip-btn');
  
  if (onboardingBtn) {
    onboardingBtn.addEventListener('click', openOnboarding);
  }
  if (skipBtn) {
    skipBtn.addEventListener('click', () => showSessionSection());
  }
}

function showSessionSection() {
  const container = document.getElementById('app-container');
  container.innerHTML = `
    <div class="session-section">
      <div class="user-header">
        <img src="${currentUser?.picture || 'icons/icon48.png'}" alt="User Avatar" class="user-avatar">
        <div class="user-info">
          <h3 id="user-name-display">${currentUser?.name || 'User'}</h3>
          <p class="user-email">${currentUser?.email || ''}</p>
        </div>
      </div>
      
      <div class="stats-grid">
        <div class="stat-item">
          <span class="stat-number" id="today-sessions">${currentUser?.sessions_today || 0}</span>
          <span class="stat-label">Today's Sessions</span>
        </div>
        <div class="stat-item">
          <span class="stat-number" id="streak-display">${currentUser?.streak || 0}</span>
          <span class="stat-label">Day Streak</span>
        </div>
        <div class="stat-item clickable" id="view-last-post">
          <span class="stat-number">📝</span>
          <span class="stat-label">View Last Post</span>
        </div>
      </div>
      
      <div class="session-controls">
        <div class="session-status">
          <div id="status-indicator" class="status-indicator"></div>
          <span id="time-display">00:00</span>
        </div>
        
        <button id="start-btn" class="action-btn primary">
          Start Building
        </button>
        
        <button id="stop-btn" class="action-btn danger hidden">
          End Session
        </button>
      </div>
      
      <div class="quick-actions">
        <button id="settings-btn" class="action-btn secondary">Settings</button>
        <button id="logout-btn" class="action-btn secondary">Logout</button>
      </div>
    </div>
  `;

  // Assign DOM elements and add event listeners
  startBtn = document.getElementById('start-btn');
  stopBtn = document.getElementById('stop-btn');
  settingsBtn = document.getElementById('settings-btn');
  const logoutBtn = document.getElementById('logout-btn');
  const viewLastPostBtn = document.getElementById('view-last-post');
  
  userNameDisplay = document.getElementById('user-name-display');
  todaySessions = document.getElementById('today-sessions');
  streakDisplay = document.getElementById('streak-display');
  screenshotCountDisplay = document.getElementById('screenshot-count-display');
  statusIndicator = document.getElementById('status-indicator');
  timeDisplay = document.getElementById('time-display');

  // Add event listeners
  if (startBtn) startBtn.addEventListener('click', startSession);
  if (stopBtn) stopBtn.addEventListener('click', stopSession);
  if (settingsBtn) settingsBtn.addEventListener('click', openSettings);
  if (logoutBtn) logoutBtn.addEventListener('click', handleLogout);
  if (viewLastPostBtn) {
    viewLastPostBtn.addEventListener('click', viewLastPost);
  }

  // Update display based on current session status
  if (sessionActive) {
    showActiveSession();
    startTimeUpdater();
  } else {
    showIdleSession();
  }
  
  updateUserStats();
}

function viewLastPost() {
  if (!currentUser?.last_generated_post) {
    showNotification('No post generated yet. Complete a session first!', 'info');
    return;
  }

  chrome.tabs.create({ 
    url: chrome.runtime.getURL(
      `post-viewer.html?post=${encodeURIComponent(currentUser.last_generated_post)}&userId=${currentUser.id}&sessionId=latest`
    )
  });
}

function updateUserStats() {
  if (!currentUser) return;
  
  if (userNameDisplay) userNameDisplay.textContent = currentUser.name || 'User';
  if (todaySessions) todaySessions.textContent = currentUser.sessions_today || 0;
  if (streakDisplay) streakDisplay.textContent = currentUser.streak || 0;
  if (screenshotCountDisplay) screenshotCountDisplay.textContent = screenshotCount;
}

function showActiveSession() {
  if (startBtn) startBtn.classList.add('hidden');
  if (stopBtn) stopBtn.classList.remove('hidden');
  if (statusIndicator) {
    statusIndicator.style.backgroundColor = '#4CAF50';
    statusIndicator.style.animation = 'pulse 2s infinite';
  }
}

function showIdleSession() {
  if (startBtn) startBtn.classList.remove('hidden');
  if (stopBtn) stopBtn.classList.add('hidden');
  if (statusIndicator) {
    statusIndicator.style.backgroundColor = '#757575';
    statusIndicator.style.animation = 'none';
  }
  if (timeDisplay) timeDisplay.textContent = '00:00';
}

function startTimeUpdater() {
  if (timeUpdateInterval) clearInterval(timeUpdateInterval);
  
  timeUpdateInterval = setInterval(async () => {
    if (sessionActive && sessionStartTime) {
      const elapsed = Math.floor((Date.now() - sessionStartTime) / 1000);
      if (timeDisplay) timeDisplay.textContent = formatTime(elapsed);
      
      // Update screenshot count from storage periodically
      try {
        const { sessionStats } = await chrome.storage.local.get(['sessionStats']);
        if (sessionStats) {
          screenshotCount = sessionStats.screenshotCount || 0;
          if (screenshotCountDisplay) screenshotCountDisplay.textContent = screenshotCount;
        }
      } catch (error) {
        console.error('Error updating session stats:', error);
      }
    }
  }, 1000);
}

function formatTime(seconds) {
  const hrs = Math.floor(seconds / 3600);
  const mins = Math.floor((seconds % 3600) / 60);
  const secs = seconds % 60;
  
  if (hrs > 0) {
    return `${hrs}:${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
  }
  return `${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
}

async function handleLogin(event) {
  event.preventDefault();
  
  const email = document.getElementById('login-email').value;
  const password = document.getElementById('login-password').value;
  const loginBtn = document.getElementById('login-btn');
  
  if (!email || !password) {
    showNotification('Please fill in all fields', 'error');
    return;
  }
  
  try {
    loginBtn.disabled = true;
    loginBtn.textContent = 'Signing in...';
    
    console.log('🔐 Starting login...');
    
    const response = await chrome.runtime.sendMessage({ 
      action: 'EMAIL_LOGIN',
      credentials: { email, password }
    });
    
    if (response.success) {
      currentUser = response.user;
      console.log('✅ Login successful:', currentUser.email);
      
      // Check if user needs onboarding
      await checkUserOnboardingStatus();
    } else {
      throw new Error(response.error || 'Login failed');
    }
    
  } catch (error) {
    console.error('❌ Login error:', error);
    showNotification(error.message || 'Login failed. Please try again.', 'error');
  } finally {
    loginBtn.disabled = false;
    loginBtn.textContent = 'Sign In';
  }
}

async function handleRegister(event) {
  event.preventDefault();
  
  const name = document.getElementById('register-name').value;
  const email = document.getElementById('register-email').value;
  const password = document.getElementById('register-password').value;
  const registerBtn = document.getElementById('register-btn');
  
  if (!name || !email || !password) {
    showNotification('Please fill in all fields', 'error');
    return;
  }
  
  if (password.length < 6) {
    showNotification('Password must be at least 6 characters', 'error');
    return;
  }
  
  try {
    registerBtn.disabled = true;
    registerBtn.textContent = 'Creating account...';
    
    console.log('📝 Starting registration...');
    
    const response = await chrome.runtime.sendMessage({ 
      action: 'EMAIL_REGISTER',
      credentials: { name, email, password }
    });
    
    if (response.success) {
      currentUser = response.user;
      console.log('✅ Registration successful:', currentUser.email);
      
      showNotification('Account created successfully!', 'success');
      
      // New users always need onboarding
      showOnboardingSection();
    } else {
      throw new Error(response.error || 'Registration failed');
    }
    
  } catch (error) {
    console.error('❌ Registration error:', error);
    showNotification(error.message || 'Registration failed. Please try again.', 'error');
  } finally {
    registerBtn.disabled = false;
    registerBtn.textContent = 'Sign Up';
  }
}

async function handleLogout() {
  try {
    showNotification('Logging out...', 'info');
    
    const response = await chrome.runtime.sendMessage({ action: 'LOGOUT' });
    
    if (response.success) {
      currentUser = null;
      sessionActive = false;
      sessionStartTime = null;
      screenshotCount = 0;
      
      if (timeUpdateInterval) {
        clearInterval(timeUpdateInterval);
        timeUpdateInterval = null;
      }
      
      showNotification('Logged out successfully', 'success');
      setTimeout(() => {
        showAuthSection();
      }, 1000);
    } else {
      throw new Error(response.error || 'Logout failed');
    }
  } catch (error) {
    console.error('❌ Logout error:', error);
    showNotification('Logout failed. Please try again.', 'error');
  }
}

async function startSession() {
  if (!currentUser) return;
  
  try {
    if (startBtn) {
      startBtn.disabled = true;
      startBtn.textContent = 'Starting...';
    }
    
    console.log('▶️ Starting session...');

    const sessionId = Date.now().toString();
    
    // Send message to background script to start session
    const response = await chrome.runtime.sendMessage({ 
      action: 'START_SESSION',
      userData: { ...currentUser, sessionId } 
    });
    
    if (response.success) {
      sessionActive = true;
      sessionStartTime = response.sessionStartTime;
      screenshotCount = 0;
      showActiveSession();
      startTimeUpdater();
      
      // Update sessions today count if provided
      if (response.sessionsToday !== undefined) {
        currentUser.sessions_today = response.sessionsToday;
        updateUserStats();
      }
      
      showNotification('🏗️ Session started! Start building...', 'success');
    } else {
      throw new Error(response.error || 'Failed to start session');
    }
    
  } catch (error) {
    console.error('❌ Error starting session:', error);
    showNotification('Failed to start session. Please try again.', 'error');
  } finally {
    if (startBtn) {
      startBtn.disabled = false;
      startBtn.textContent = 'Start Building';
    }
  }
}

async function stopSession() {
  if (!currentUser) return;
  
  try {
    // Show feedback form before stopping session
    const feedback = await showFeedbackForm();
    if (stopBtn) {
      stopBtn.disabled = true;
      stopBtn.textContent = 'Generating post...';
    }
    
    console.log('⏹️ Stopping session...');
    
    // Send message to background script to stop session
    const response = await chrome.runtime.sendMessage({ 
      action: 'STOP_SESSION',
      userData: { 
        ...currentUser, 
        feedback: feedback  // Add feedback to userData
      }
    });
    
    if (response.success) {
      sessionActive = false;
      sessionStartTime = null;
      showIdleSession();
      
      if (timeUpdateInterval) {
        clearInterval(timeUpdateInterval);
        timeUpdateInterval = null;
      }
      
      // Show completion message with generated post
      showSessionComplete(response);
    } else {
      throw new Error(response.error || 'Failed to stop session');
    }
    
  } catch (error) {
    console.error('❌ Error stopping session:', error);
    showNotification('Failed to end session. Please try again.', 'error');
  } finally {
    if (stopBtn) {
      stopBtn.disabled = false;
      stopBtn.textContent = 'End Session';
    }
  }
}

function showSessionComplete(sessionData) {
  const minutes = Math.floor(sessionData.sessionDuration / 60);
  const screenshots = sessionData.screenshotCount || 0;
  const screenshotUrls = sessionData.screenshot_urls || [];
  
  // Create completion modal
  const modal = document.createElement('div');
  modal.style.cssText = `
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0,0,0,0.8);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10000;
    backdrop-filter: blur(5px);
  `;
  
  const content = document.createElement('div');
  content.style.cssText = `
    background: white;
    border-radius: 16px;
    padding: 24px;
    max-width: 500px;
    margin: 20px;
    text-align: center;
    box-shadow: 0 20px 40px rgba(0,0,0,0.3);
    max-height: 80vh;
    overflow-y: auto;
  `;

  const screenshotPreview = screenshotUrls.length > 0 ? `
    <div style="margin: 16px 0;">
      <h4 style="margin: 0 0 8px 0; font-size: 14px; color: #374151;">Screenshots captured:</h4>
      <div style="display: flex; gap: 8px; justify-content: center; flex-wrap: wrap;">
        ${screenshotUrls.slice(0, 4).map(url => `
          <img src="${url}" style="width: 60px; height: 40px; object-fit: cover; border-radius: 4px; border: 1px solid #e5e7eb;">
        `).join('')}
        ${screenshotUrls.length > 4 ? `<span style="font-size: 12px; color: #6b7280;">+${screenshotUrls.length - 4} more</span>` : ''}
      </div>
    </div>
  ` : '';
  
  content.innerHTML = `
    <div style="font-size: 48px; margin-bottom: 16px;">🎉</div>
    <h2 style="margin: 0 0 12px 0; color: #1f2937;">Session Complete!</h2>
    <p style="color: #6b7280; margin: 0 0 16px 0;">
      ${minutes} minutes • ${screenshots} screenshots captured
    </p>
    ${screenshotPreview}
    <div style="background: #f3f4f6; padding: 16px; border-radius: 12px; margin: 20px 0; text-align: left; max-height: 200px; overflow-y: auto;">
      <h3 style="margin: 0 0 8px 0; font-size: 14px; color: #374151;">Generated Post:</h3>
      <p style="font-size: 14px; line-height: 1.5; color: #1f2937; margin: 0; white-space: pre-wrap;">
        ${sessionData.post || 'Post generation in progress...'}
      </p>
    </div>
    <div style="display: flex; gap: 12px; justify-content: center;">
      <button id="viewFullPost" style="
        background: #10b981;
        color: white;
        border: none;
        padding: 12px 24px;
        border-radius: 8px;
        cursor: pointer;
        font-weight: 600;
      ">View Full Post</button>
      <button id="copyPost" style="
        background: #667eea;
        color: white;
        border: none;
        padding: 12px 24px;
        border-radius: 8px;
        cursor: pointer;
        font-weight: 600;
      ">Copy Post</button>
      <button id="closeModal" style="
        background: #f3f4f6;
        color: #374151;
        border: none;
        padding: 12px 24px;
        border-radius: 8px;
        cursor: pointer;
        font-weight: 600;
      ">Close</button>
    </div>
  `;
  
  modal.appendChild(content);
  document.body.appendChild(modal);
  
  // Add event handlers
  const viewFullBtn = content.querySelector('#viewFullPost');
  const copyBtn = content.querySelector('#copyPost');
  const closeBtn = content.querySelector('#closeModal');
  
  viewFullBtn.addEventListener('click', () => {
    viewLastPost(); // This will open the enhanced viewer
    modal.remove();
  });
  
  copyBtn.addEventListener('click', async () => {
    try {
      await navigator.clipboard.writeText(sessionData.post || '');
      copyBtn.textContent = 'Copied!';
      copyBtn.style.background = '#4ade80';
      setTimeout(() => {
        copyBtn.textContent = 'Copy Post';
        copyBtn.style.background = '#667eea';
      }, 2000);
    } catch (error) {
      console.error('Failed to copy:', error);
      showNotification('Failed to copy post', 'error');
    }
  });
  
  closeBtn.addEventListener('click', () => {
    modal.remove();
  });
}

function showFeedbackForm() {
  return new Promise((resolve) => {
    const modal = document.createElement('div');
    modal.style.cssText = `
      position: fixed;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      background: rgba(0,0,0,0.8);
      display: flex;
      align-items: center;
      justify-content: center;
      z-index: 10000;
      backdrop-filter: blur(5px);
    `;
    
    const content = document.createElement('div');
    content.style.cssText = `
      background: white;
      border-radius: 16px;
      padding: 24px;
      max-width: 400px;
      margin: 20px;
      box-shadow: 0 20px 40px rgba(0,0,0,0.3);
    `;
    
    content.innerHTML = `
      <h3 style="margin: 0 0 16px 0; color: #1f2937;">Help us improve your post</h3>
      <p style="color: #6b7280; margin: 0 0 16px 0; font-size: 14px;">
        Tell us what you'd like to highlight or any context that would make your post more accurate.
      </p>
      <textarea 
        id="feedback-input" 
        placeholder="e.g., I was working on a React component, learning about state management, debugging API calls..."
        style="
          width: 100%;
          height: 100px;
          border: 2px solid #e5e7eb;
          border-radius: 8px;
          padding: 12px;
          font-size: 14px;
          resize: vertical;
          font-family: inherit;
          box-sizing: border-box;
        "
      ></textarea>
      <div style="display: flex; gap: 12px; justify-content: flex-end; margin-top: 16px;">
        <button id="skip-feedback" style="
          background: #f3f4f6;
          color: #374151;
          border: none;
          padding: 8px 16px;
          border-radius: 6px;
          cursor: pointer;
          font-size: 14px;
        ">Skip</button>
        <button id="submit-feedback" style="
          background: #667eea;
          color: white;
          border: none;
          padding: 8px 16px;
          border-radius: 6px;
          cursor: pointer;
          font-size: 14px;
        ">Continue</button>
      </div>
    `;
    
    modal.appendChild(content);
    document.body.appendChild(modal);
    
    const feedbackInput = content.querySelector('#feedback-input');
    const submitBtn = content.querySelector('#submit-feedback');
    const skipBtn = content.querySelector('#skip-feedback');
    
    feedbackInput.focus();
    
    submitBtn.addEventListener('click', () => {
      const feedback = feedbackInput.value.trim();
      modal.remove();
      resolve(feedback);
    });
    
    skipBtn.addEventListener('click', () => {
      modal.remove();
      resolve('');
    });
    
    // Submit on Enter (Ctrl/Cmd + Enter)
    feedbackInput.addEventListener('keydown', (e) => {
      if (e.key === 'Enter' && (e.ctrlKey || e.metaKey)) {
        e.preventDefault();
        submitBtn.click();
      }
    });
  });
}

function showNotification(message, type = 'info') {
  const notification = document.createElement('div');
  const bgColor = type === 'error' ? '#f87171' : type === 'success' ? '#4ade80' : '#60a5fa';
  
  notification.style.cssText = `
    position: fixed;
    top: 10px;
    left: 10px;
    right: 10px;
    background: ${bgColor};
    color: white;
    padding: 12px 16px;
    border-radius: 8px;
    text-align: center;
    font-size: 14px;
    font-weight: 500;
    z-index: 1000;
    box-shadow: 0 4px 12px rgba(0,0,0,0.2);
    transform: translateY(-100%);
    transition: transform 0.3s ease;
  `;
  notification.textContent = message;
  
  document.body.appendChild(notification);
  
  // Animate in
  setTimeout(() => {
    notification.style.transform = 'translateY(0)';
  }, 10);
  
  // Animate out and remove
  setTimeout(() => {
    notification.style.transform = 'translateY(-100%)';
    setTimeout(() => {
      if (notification.parentNode) {
        notification.remove();
      }
    }, 300);
  }, 4000);
}

function openOnboarding() {
  chrome.runtime.sendMessage({ action: 'OPEN_ONBOARDING' });
}

function openSettings() {
  chrome.runtime.sendMessage({ action: 'OPEN_SETTINGS' });
}

async function loadUserData() {
  try {
    // Check chrome storage for user data
    const result = await chrome.storage.local.get(['user']);
    
    if (result.user) {
      currentUser = result.user;
      console.log('✅ User loaded from storage:', currentUser.email);
      
      // Check if user has completed onboarding
      if (currentUser.preferences) {
        showSessionSection();
        updateUserStats();
      } else {
        showOnboardingSection();
      }
    } else {
      console.log('ℹ️ No user found, showing auth');
      showAuthSection();
    }
  } catch (error) {
    console.error('❌ Error loading user data:', error);
    showAuthSection();
  }
}
