// Onboarding state
let currentStep = 1;
const totalSteps = 3;
const preferences = {
  postType: null,
  tone: null,
  skillFocus: '',
  linkedin_handle: null,
  twitter_handle: null
};

// DOM elements
const progressFill = document.getElementById('progressFill');
const stepIndicator = document.getElementById('stepIndicator');
const backBtn = document.getElementById('backBtn');
const nextBtn = document.getElementById('nextBtn');
const navigation = document.getElementById('navigation');
const loading = document.getElementById('loading');
const success = document.getElementById('success');

// Initialize
document.addEventListener('DOMContentLoaded', () => {
  console.log('DOM loaded, initializing onboarding...');
  setupEventListeners();
  updateUI();
});

// Fallback if DOMContentLoaded already fired
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initialize);
} else {
  initialize();
}

function initialize() {
  console.log('Initializing onboarding...');
  setupEventListeners();
  updateUI();
}

function setupEventListeners() {
  console.log('Setting up event listeners...');
  // Option selection
  document.addEventListener('click', (e) => {
    const option = e.target.closest('.option');
    if (!option) return;
    
    const step = option.closest('.step');
    const stepId = step.id;
    
    // Clear previous selections in this step
    step.querySelectorAll('.option').forEach(opt => opt.classList.remove('selected'));
    
    // Select current option
    option.classList.add('selected');
    
    // Store preference
    const value = option.dataset.value;
    if (stepId === 'step1') {
      preferences.postType = value;
    } else if (stepId === 'step2') {
      preferences.tone = value;
    }
    
    updateUI();
  });

  // Text inputs
  document.getElementById('skillFocus').addEventListener('input', (e) => {
    preferences.skillFocus = e.target.value;
    updateUI();
  });

  document.getElementById('linkedinHandle').addEventListener('input', (e) => {
    preferences.linkedin_handle = e.target.value || null;
  });

  document.getElementById('twitterHandle').addEventListener('input', (e) => {
    preferences.twitter_handle = e.target.value || null;
  });

  // Navigation
  backBtn.addEventListener('click', (e) => {
    e.preventDefault();
    if (currentStep > 1) {
      currentStep--;
      updateUI();
    }
  });

  nextBtn.addEventListener('click', (e) => {
    e.preventDefault();
    if (nextBtn.disabled) return; // Prevent action if disabled
    
    if (currentStep < totalSteps) {
      currentStep++;
      updateUI();
    } else {
      completeOnboarding();
    }
  });
}

function updateUI() {
  // Update progress bar
  const progress = (currentStep / totalSteps) * 100;
  progressFill.style.width = `${progress}%`;

  // Update step indicator
  stepIndicator.textContent = `Step ${currentStep} of ${totalSteps}`;

  // Show current step
  document.querySelectorAll('.step').forEach((step, index) => {
    step.classList.toggle('active', index + 1 === currentStep);
  });

  // Update navigation buttons
  backBtn.disabled = currentStep === 1;
  
  // Update next button
  let canProceed = false;
  if (currentStep === 1) {
    canProceed = !!preferences.postType;
  } else if (currentStep === 2) {
    canProceed = !!preferences.tone;
  } else if (currentStep === 3) {
    const words = preferences.skillFocus.trim().split(/\s+/).filter(word => word.length > 0);
    canProceed = words.length >= 10;
  }

  nextBtn.disabled = !canProceed;
  nextBtn.textContent = currentStep === totalSteps ? 'Complete Setup' : 'Next';
}

async function completeOnboarding() {
  try {
    // Show loading
    document.querySelector('.step-content').style.display = 'none';
    navigation.style.display = 'none';
    loading.style.display = 'block';

    // Get current user from extension storage
    const userResult = await chrome.storage.local.get(['user']);
    if (!userResult.user) {
      throw new Error('No user found');
    }

    // Prepare preferences for API
    const apiPreferences = {
      user_id: userResult.user.id,
      content_type: preferences.postType,
      writing_tone: preferences.tone,
      journey_stage: preferences.skillFocus,
      linkedin_handle: preferences.linkedin_handle,
      twitter_handle: preferences.twitter_handle
    };

    console.log('Saving preferences:', apiPreferences);

    // Send to background script to save preferences
    const response = await chrome.runtime.sendMessage({
      action: 'SAVE_PREFERENCES',
      preferences: apiPreferences
    });

    if (!response.success) {
      throw new Error(response.error || 'Failed to save preferences');
    }

    // Show success
    loading.style.display = 'none';
    success.style.display = 'block';

    // Auto-close after 3 seconds
    setTimeout(() => {
      window.close();
    }, 3000);

  } catch (error) {
    console.error('Onboarding error:', error);
    alert('Failed to complete setup. Please try again.');
    
    // Reset UI
    loading.style.display = 'none';
    document.querySelector('.step-content').style.display = 'block';
    navigation.style.display = 'flex';
  }
}