// Background script for Hudddle Extension
let sessionActive = false;
let screenshotInterval;
let sessionStartTime;
let screenshotCount = 0;
let isSavingPreferences = false;

// API Configuration
const API_BASE_URL = 'https://hudddle-post-feature.onrender.com';
const LOCAL_API_URL = 'http://127.0.0.1:8000';

// Get API URL (prefer local for development)
function getApiUrl() {
  return API_BASE_URL;
}

// Listen for extension installation
chrome.runtime.onInstalled.addListener(() => {
  console.log('🏗️ Hudddle extension installed');
  // Initialize storage
  chrome.storage.local.set({ 
    sessionActive: false,
    user: null,
    sessionStats: {
      screenshotCount: 0,
      sessionStartTime: null
    }
  });
});

// Handle messages from content script and popup
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  switch (message.action) {
    case 'EMAIL_REGISTER':
      handleEmailRegister(message.credentials, sendResponse);
      break;

    case 'EMAIL_LOGIN':
      handleEmailLogin(message.credentials, sendResponse);
      break;

    case 'LOGOUT':
      handleLogout(sendResponse);
      break;

    case 'CHECK_AUTH':
      checkAuthStatus(sendResponse);
      break;
    
    case 'FETCH_USER':
      fetchUserFromBackend(message.userId, sendResponse);
      break;
    
    case 'SAVE_PREFERENCES':
      saveUserPreferences(message.preferences, sendResponse);
      break;
    
    case 'START_SESSION':
      startSession(message.userData, sendResponse);
      break;
    
    case 'STOP_SESSION':
      stopSession(message.userData, sendResponse);
      break;
    
    case 'GET_SESSION_STATUS':
      getSessionStatus(sendResponse);
      break;
    
    case 'OPEN_ONBOARDING':
      openOnboardingPage();
      break;
    
    case 'OPEN_SETTINGS':
      openSettingsPage();
      break;

    case 'UPDATE_USER':
      updateUserData(message.userData, sendResponse);
      break;
  }
  return true; // Keep message channel open for async response
});

async function handleEmailRegister(credentials, sendResponse) {
  try {
    console.log('📝 Registering user...');
    
    const authResponse = await registerWithBackend({
      email: credentials.email,
      password: credentials.password,
      name: credentials.name
    });
    
    console.log('✅ User registered:', authResponse.email);
    
    // Store user data and token securely
    await chrome.storage.local.set({ 
      user: authResponse,
      authToken: authResponse.access_token,
      authTimestamp: Date.now()
    });
    
    sendResponse({ 
      success: true, 
      user: authResponse,
      token: authResponse.access_token
    });
    
  } catch (error) {
    console.error('❌ Registration error:', error);
    
    // Clean up any partial auth state
    await chrome.storage.local.remove(['user', 'authToken', 'authTimestamp']);
    
    sendResponse({ 
      success: false, 
      error: error.message
    });
  }
}

async function handleEmailLogin(credentials, sendResponse) {
  try {
    console.log('🔐 Logging in user...');
    
    const authResponse = await loginWithBackend({
      email: credentials.email,
      password: credentials.password
    });
    
    console.log('✅ User logged in:', authResponse.email);
    
    // Store user data and token securely
    await chrome.storage.local.set({ 
      user: authResponse,
      authToken: authResponse.access_token,
      authTimestamp: Date.now()
    });
    
    sendResponse({ 
      success: true, 
      user: authResponse,
      token: authResponse.access_token
    });
    
  } catch (error) {
    console.error('❌ Login error:', error);
    
    sendResponse({ 
      success: false, 
      error: error.message
    });
  }
}

async function registerWithBackend(userData) {
  const apiUrl = getApiUrl();
  
  try {
    const response = await fetch(`${apiUrl}/auth/register`, {
      method: 'POST',
      headers: { 
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(userData)
    });
    
    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      throw new Error(errorData.detail || `Registration failed: ${response.status}`);
    }
    
    return await response.json();
  } catch (error) {
    console.error('Backend registration error:', error);
    throw error;
  }
}

async function loginWithBackend(credentials) {
  const apiUrl = getApiUrl();
  
  try {
    const response = await fetch(`${apiUrl}/auth/login`, {
      method: 'POST',
      headers: { 
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(credentials)
    });
    
    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      throw new Error(errorData.detail || `Login failed: ${response.status}`);
    }
    
    return await response.json();
  } catch (error) {
    console.error('Backend login error:', error);
    throw error;
  }
}

// Handle logout
async function handleLogout(sendResponse) {
  try {
    console.log('🚪 Logging out user...');
    
    // Get stored token
    const result = await chrome.storage.local.get(['authToken']);
    
    // Revoke token if it exists
    if (result.authToken) {
      // Revoke the token with Google
      await fetch(`https://oauth2.googleapis.com/revoke?token=${result.authToken}`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded'
        }
      }).catch(error => {
        console.warn('Failed to revoke Google token:', error);
      });
      
      // Remove from Chrome's cache
      chrome.identity.removeCachedAuthToken({ 
        token: result.authToken 
      });
    }
    
    // Clear all stored auth data
    await chrome.storage.local.remove(['user', 'authToken', 'authTimestamp']);
    
    // Stop any active session
    if (sessionActive) {
      await stopSession(null, () => {});
    }
    
    console.log('✅ User logged out successfully');
    sendResponse({ success: true });
    
  } catch (error) {
    console.error('❌ Logout error:', error);
    sendResponse({ success: false, error: error.message });
  }
}

// Check authentication status
async function checkAuthStatus(sendResponse) {
  try {
    const result = await chrome.storage.local.get(['user', 'authToken', 'authTimestamp']);
    
    const isAuthenticated = !!(result.user && result.authToken);
    const tokenAge = result.authTimestamp ? Date.now() - result.authTimestamp : 0;
    const tokenExpired = tokenAge > (60 * 60 * 1000); // 1 hour
    
    if (isAuthenticated && !tokenExpired) {
      sendResponse({ 
        success: true, 
        isAuthenticated: true,
        user: result.user 
      });
    } else if (tokenExpired) {
      // Token expired, clear auth data
      await chrome.storage.local.remove(['user', 'authToken', 'authTimestamp']);
      sendResponse({ 
        success: true, 
        isAuthenticated: false,
        reason: 'Token expired'
      });
    } else {
      sendResponse({ 
        success: true, 
        isAuthenticated: false 
      });
    }
    
  } catch (error) {
    console.error('Error checking auth status:', error);
    sendResponse({ success: false, error: error.message });
  }
}

// Fetch user data from backend
async function fetchUserFromBackend(userId, sendResponse) {
  try {
    const apiUrl = getApiUrl();
    const response = await fetch(`${apiUrl}/user/${userId}`);
    
    if (response.status === 401) {
      // Token invalid, trigger re-authentication
      await chrome.storage.local.remove(['user', 'authToken', 'authTimestamp']);
      throw new Error('Authentication expired. Please log in again.');
    }
    
    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      throw new Error(`Failed to fetch user: ${response.status} ${response.statusText}`);
    }
    
    const userData = await response.json();
    
    // Update stored user data
    await chrome.storage.local.set({ user: userData });
    
    sendResponse({ success: true, user: userData });
  } catch (error) {
    console.error('Failed to fetch user:', error);
    sendResponse({ success: false, error: error.message });
  }
}

// Save user preferences
async function saveUserPreferences(preferences, sendResponse) {
  if (isSavingPreferences) {
    console.log('Preferences save already in progress, skipping duplicate call');
    return;
  }
  try {
    isSavingPreferences = true;

    const apiUrl = getApiUrl();
    const response = await fetch(`${apiUrl}/preferences`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(preferences)
    });
    
    if (response.status === 401) {
      await chrome.storage.local.remove(['user', 'authToken', 'authTimestamp']);
      throw new Error('Authentication expired. Please log in again.');
    }
    
    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      throw new Error(`Failed to save preferences: ${response.status} - ${errorData.details || response.statusText}`);
    }
    
    // Fetch updated user data
    const updatedUserResponse = await fetch(`${apiUrl}/user/${preferences.user_id}`);
    const updatedUser = await updatedUserResponse.json();
    
    // Update stored user data
    await chrome.storage.local.set({ user: updatedUser });
    
    sendResponse({ success: true, user: updatedUser });
  } catch (error) {
    console.error('Failed to save preferences:', error);
    sendResponse({ success: false, error: error.message });
  } finally {
    isSavingPreferences = false;
  }
}

// Start session
async function startSession(userData, sendResponse) {
  try {
    console.log('▶️ Starting session for user:', userData.id);
    
    sessionActive = true;
    sessionStartTime = Date.now();
    screenshotCount = 0;

    const sessionId = `session_${Date.now()}_${userData.id}`;
    
    // Update storage
    await chrome.storage.local.set({ 
      sessionActive: true,
      sessionStartTime: sessionStartTime,
      sessionId: sessionId,
      sessionUser: userData,
      sessionStats: {
        screenshotCount: 0,
        sessionStartTime
      }
    });
    
    // Notify backend
    const apiUrl = getApiUrl();
    const response = await fetch(`${apiUrl}/session/start`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ user_id: userData.id })
    });
    
    if (response.ok) {
      const result = await response.json();
      console.log('✅ Session started on backend, sessions today:', result.sessions_today);
      startScreenshotCapture(userData);
      
      // Update extension badge
      chrome.action.setBadgeText({ text: '●' });
      chrome.action.setBadgeBackgroundColor({ color: '#4CAF50' });
      
      sendResponse({ success: true, sessionStartTime, sessionsToday: result.sessions_today });
    } else {
      const errorData = await response.json().catch(() => ({}));
      throw new Error(errorData.detail || 'Failed to start session on backend');
    }
  } catch (error) {
    console.error('❌ Failed to start session:', error);
    sessionActive = false;
    await chrome.storage.local.remove(['sessionActive', 'sessionStartTime', 'sessionUser']);
    sendResponse({ success: false, error: error.message });
  }
}

// Stop session and generate post
async function stopSession(userData, sendResponse) {
  try {
    console.log('⏹️ Stopping session...');
    
    sessionActive = false;
    
    if (screenshotInterval) {
      clearInterval(screenshotInterval);
      screenshotInterval = null;
    }
    
    // Calculate session duration
    const sessionDuration = sessionStartTime ? 
      Math.floor((Date.now() - sessionStartTime) / 1000) : 0;
    
    // Clear badge
    chrome.action.setBadgeText({ text: '' });

    const { sessionId } = await chrome.storage.local.get(['sessionId']);
    
    // Clear session storage
    await chrome.storage.local.remove(['sessionActive', 'sessionStartTime', 'sessionUser', 'sessionId']);
    
    // Update storage with inactive state
    await chrome.storage.local.set({ 
      sessionActive: false,
      sessionStats: {
        screenshotCount: 0,
        sessionStartTime: null
      }
    });
    
    // End session on backend and generate post
    if (userData) {
      const apiUrl = getApiUrl();

      const requestPayload = {
        user_id: userData.id,
        feedback: userData.feedback || '',
        session_id: sessionId || sessionStartTime?.toString()
      };
      
      console.log('Sending session end request:', requestPayload);

      const response = await fetch(`${apiUrl}/session/end`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(requestPayload)
      });
      
      if (response.ok) {
        const result = await response.json();
        console.log('✅ Session ended, post generated');

        await chrome.storage.local.set({ 
          lastCompletedSessionId: sessionId || sessionStartTime?.toString() 
        });
        
        sendResponse({ 
          success: true, 
          post: result.post,
          screenshot_urls: result.screenshot_urls || [],
          sessionDuration,
          screenshotCount,
          sessionId: sessionId || sessionStartTime?.toString()
        });
      } else {
        const errorData = await response.json().catch(() => ({}));
        console.error('Backend error response:', errorData);
        throw new Error(errorData.detail || 'Failed to end session on backend');
      }
    } else {
      sendResponse({ success: true });
    }
  } catch (error) {
    console.error('❌ Failed to stop session:', error);
    sendResponse({ success: false, error: error.message });
  }
}

// Get current session status
async function getSessionStatus(sendResponse) {
  const { sessionActive, sessionStartTime, sessionUser, sessionStats, sessionId } = 
    await chrome.storage.local.get(['sessionActive', 'sessionStartTime', 'sessionUser', 'sessionStats', 'sessionId']);
  
  sendResponse({
    active: sessionActive || false,
    startTime: sessionStartTime || null,
    user: sessionUser || null,
    screenshotCount: sessionStats?.screenshotCount || 0,
    sessionId: sessionId || null
  });
}

// Screenshot capture functionality
function startScreenshotCapture(userData) {
  console.log('📸 Starting screenshot capture every 5 mintues');
  
  screenshotInterval = setInterval(() => {
    if (sessionActive) {
      captureScreenshot(userData);
    }
  // }, 300000); // Every 5 mintues
  }, 60000);
}

async function captureScreenshot(userData) {
  try {
    console.log('📸 Attempting to capture screenshot...');

    // Get session ID from storage:
    const { sessionId } = await chrome.storage.local.get(['sessionId']);

    // Get the current active tab
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

    if (!tab) {
      console.log('❌ No active tab found');
      return;
    }
    
    console.log('Current tab:', tab.url);
    
    // Skip chrome://, chrome-extension://, and other restricted URLs
    if (tab.url && !tab.url.startsWith('chrome://') && 
        !tab.url.startsWith('chrome-extension://') &&
        !tab.url.startsWith('about:') &&
        !tab.url.startsWith('file://')) {
      
      chrome.tabs.captureVisibleTab(tab.windowId, { format: 'png' }, async (dataUrl) => {
        if (chrome.runtime.lastError) {
          console.error('❌ Screenshot capture failed:', chrome.runtime.lastError);
          return;
        }
        
        console.log('✅ Screenshot captured, processing...');
        
        try {
          // Convert data URL directly to base64 without fetch
          const base64Data = dataUrl.split(',')[1];
          
          console.log('✅ Blob created, uploading...');
          
          // Upload to backend
          const success = await uploadScreenshotBase64(base64Data, userData, sessionId);
          if (success) {
            console.log('✅ Screenshot uploaded successfully');
          } else {
            console.log('❌ Screenshot upload failed');
          }
        } catch (error) {
          console.error('❌ Error processing screenshot:', error);
        }
      });
    } else {
      console.log('⏭️ Skipping screenshot - invalid tab URL:', tab.url);
    }
  } catch (error) {
    console.error('❌ Screenshot capture error:', error);
  }
}

async function uploadScreenshotBase64(base64Data, userData, sessionId) {
  try {
    console.log('📤 Uploading to backend...');
    
    const apiUrl = getApiUrl();
    const response = await fetch(`${apiUrl}/upload-screenshot-base64`, {
      method: 'POST',
      headers: { 
        'Content-Type': 'application/json' 
      },
      body: JSON.stringify({
        screenshot: base64Data, // Just the base64 part without data: prefix
        user_id: userData.id,
        timestamp: new Date().toISOString(),
        session_id: sessionId
      })
    });
    
    if (response.ok) {
      const result = await response.json();
      screenshotCount++;
      
      // Update screenshot count in storage
      const { sessionStats } = await chrome.storage.local.get(['sessionStats']);
      await chrome.storage.local.set({ 
        sessionStats: {
          ...sessionStats,
          screenshotCount
        }
      });
      
      console.log(`✅ Screenshot ${screenshotCount} uploaded:`, result.image_url);
      return true;
    } else {
      const errorText = await response.text();
      console.error('❌ Upload failed:', response.status, errorText);
      return false;
    }
  } catch (error) {
    console.error('❌ Upload error:', error);
    return false;
  }
}

async function updateUserData(userData, sendResponse) {
  try {
    const apiUrl = getApiUrl();
    const response = await fetch(`${apiUrl}/users/${userData.id}`, {
      method: 'PATCH',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(userData)
    });
    
    if (response.status === 401) {
      await chrome.storage.local.remove(['user', 'authToken', 'authTimestamp']);
      throw new Error('Authentication expired. Please log in again.');
    }
    
    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      throw new Error(`Failed to update user: ${response.status} - ${errorData.details || response.statusText}`);
    }
    
    // Fetch updated user data
    const updatedUserResponse = await fetch(`${apiUrl}/user/${userData.id}`);
    const updatedUser = await updatedUserResponse.json();
    
    // Update stored user data
    await chrome.storage.local.set({ user: updatedUser });
    
    sendResponse({ success: true, user: updatedUser });
  } catch (error) {
    console.error('Failed to update user:', error);
    sendResponse({ success: false, error: error.message });
  }
}

// Helper function to convert blob to base64
function blobToBase64(blob) {
  return new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.onload = () => resolve(reader.result);
    reader.onerror = reject;
    reader.readAsDataURL(blob);
  });
}

// Open onboarding page
function openOnboardingPage() {
  chrome.tabs.create({ 
    url: chrome.runtime.getURL('onboarding.html') 
  });
}

// Open settings page
function openSettingsPage() {
  chrome.tabs.create({ 
    url: chrome.runtime.getURL('settings.html') 
  });
}

// Periodic sync to keep session state consistent
setInterval(async () => {
  const { sessionActive } = await chrome.storage.local.get(['sessionActive']);
  if (sessionActive !== (sessionActive || false)) {
    // Sync state if mismatch
    await chrome.storage.local.set({ sessionActive });
  }
}, 5000);
